#include <windows.h>

#include <stdio.h>

#include ".\level.h"
#include ".\GameObject.h"
#include ".\LightAndDark.h"
#include ".\Player.h"
#include ".\Lamp.h"
#include ".\Guard.h"



CLevel::CLevel() :
  m_iWidth( 0 ),
  m_iHeight( 0 ),
  m_pData( NULL ),
  m_pTileInfo( NULL ),
  m_fOffsetX( 0.0f ),
  m_fOffsetY( 0.0f ),
  m_bFullyLit( false ),
  m_bPlayerHasBeenCaught( false ),
  m_bLevelDone( false ),
  m_CaughtReason( Light::CR_INVALID )
{
}

CLevel::~CLevel()
{

  Reset();

}



void CLevel::Reset()
{

  tListObjects::iterator    it( m_listObjects.begin() );
  while ( it != m_listObjects.end() )
  {
    delete *it;

    ++it;
  }
  m_listObjects.clear();

  if ( m_pData )
  {
    delete[] m_pData;
    m_pData = NULL;
  }
  if ( m_pTileInfo )
  {
    delete[] m_pTileInfo;
    m_pTileInfo = NULL;
  }
  m_iWidth = 0;
  m_iHeight = 0;
  m_fOffsetX = 0.0f;
  m_fOffsetY = 0.0f;

  m_bPlayerHasBeenCaught  = false;
  m_bLevelDone            = false;
  m_bFullyLit             = false;

  m_CaughtReason          = Light::CR_INVALID;

}


void CLevel::Initialize( int iWidth, int iHeight )
{

  Reset();
  m_pData     = new Light::eTextureSections[iWidth * iHeight];
  m_pTileInfo = new tTileInfo[iWidth * iHeight];

  m_iWidth  = iWidth;
  m_iHeight = iHeight;

  for ( int i = 0; i < iWidth; ++i )
  {
    for ( int j = 0; j < iHeight; ++j )
    {
      SetField( i, j, Light::TS_EMPTY );
      GetTileInfo( i, j ) = tTileInfo();
    }
  }
  m_fOffsetX = 0.0f;
  m_fOffsetY = 0.0f;

}



Light::eTextureSections CLevel::GetField( int iX, int iY )
{

  if ( m_pData == NULL )
  {
    return Light::TS_EMPTY;
  }
  if ( ( iX < 0 )
  ||   ( iX >= m_iWidth )
  ||   ( iY < 0 )
  ||   ( iY >= m_iHeight ) )
  {
    return Light::TS_EMPTY;
  }
  return m_pData[iX + m_iWidth * iY];

}



CLevel::tTileInfo& CLevel::GetTileInfo( int iX, int iY )
{

  static tTileInfo    ttDummy;

  if ( m_pData == NULL )
  {
    return ttDummy;
  }
  if ( ( iX < 0 )
  ||   ( iX >= m_iWidth )
  ||   ( iY < 0 )
  ||   ( iY >= m_iHeight ) )
  {
    return ttDummy;
  }
  return m_pTileInfo[iX + m_iWidth * iY];

}



void CLevel::SetField( int iX, int iY, Light::eTextureSections fType )
{

  if ( m_pData == NULL )
  {
    return;
  }
  if ( ( iX < 0 )
  ||   ( iX >= m_iWidth )
  ||   ( iY < 0 )
  ||   ( iY >= m_iHeight ) )
  {
    return;
  }
  m_pData[iX + m_iWidth * iY] = fType;

}



void CLevel::Display( IDirect3DDevice8* pDevice )
{

  // Schattenwurf
  // TODO - fixe Elemente vorberechnen!
  CalcShadows();

  for ( int i = 0; i < 640 / 32 + 2; ++i )
  {
    for ( int j = 0; j < 480 / 32 + 2; ++j )
    {
      tTileInfo&    TileInfo = GetTileInfo( (int)m_fOffsetX / 32 + i, (int)m_fOffsetY / 32 + j );

      theApp.RenderTextureSection( i * 32 - ( (int)m_fOffsetX ) % 32,
                                   j * 32 - ( (int)m_fOffsetY ) % 32,
                                   theApp.m_TexSec[GetField( (int)m_fOffsetX / 32 + i, (int)m_fOffsetY / 32 + j )],
                                   TileInfo.dwColor[0],
                                   TileInfo.dwColor[1],
                                   TileInfo.dwColor[2],
                                   TileInfo.dwColor[3] );
    }
  }

  tListObjects::iterator    it( m_listObjects.begin() );
  while ( it != m_listObjects.end() )
  {
    CGameObject*    pObj = *it;

    pObj->Display( pDevice, *this );

    ++it;
  }

}



void CLevel::Scroll( float fX, float fY )
{

  m_fOffsetX += fX;
  m_fOffsetY += fY;

  if ( m_fOffsetX < 0.0f )
  {
    m_fOffsetX = 0.0f;
  }
  if ( m_fOffsetX > m_iWidth * 32.0f - 640.0f )
  {
    m_fOffsetX = m_iWidth * 32.0f - 640.0f;
  }
  if ( m_fOffsetY < 0.0f )
  {
    m_fOffsetY = 0.0f;
  }
  if ( m_fOffsetY > m_iHeight * 32.0f - 480.0f )
  {
    m_fOffsetY = m_iHeight * 32.0f - 480.0f;
  }

}



void CLevel::ScrollTo( float fX, float fY )
{

  m_fOffsetX = fX;
  m_fOffsetY = fY;

  if ( m_fOffsetX < 0.0f )
  {
    m_fOffsetX = 0.0f;
  }
  if ( m_fOffsetX > m_iWidth * 32.0f - 640.0f )
  {
    m_fOffsetX = m_iWidth * 32.0f - 640.0f;
  }
  if ( m_fOffsetY < 0.0f )
  {
    m_fOffsetY = 0.0f;
  }
  if ( m_fOffsetY > m_iHeight * 32.0f - 480.0f )
  {
    m_fOffsetY = m_iHeight * 32.0f - 480.0f;
  }

}



void CLevel::Save( DWORD dwNr )
{

  char      szFileName[MAX_PATH];

  wsprintf( szFileName, "data\\level%d.dat", dwNr );

  FILE*     fileLevel = fopen( szFileName, "wb" );

  if ( fileLevel == NULL )
  {
    return;
  }
  fwrite( &m_iWidth, sizeof( m_iWidth ), 1, fileLevel );
  fwrite( &m_iHeight, sizeof( m_iHeight ), 1, fileLevel );
  fwrite( m_pData, m_iWidth * m_iHeight * sizeof( Light::eTextureSections ), 1, fileLevel );

  size_t    iCount = m_listObjects.size();
  fwrite( &iCount, sizeof( iCount ), 1, fileLevel );
  tListObjects::iterator    it( m_listObjects.begin() );
  while ( it != m_listObjects.end() )
  {
    CGameObject*    pObj = *it;

    fwrite( &pObj->m_Type, sizeof( Light::eGameObjectType ), 1, fileLevel );
    fwrite( &pObj->m_vectPosition.x, sizeof( float ), 1, fileLevel );
    fwrite( &pObj->m_vectPosition.y, sizeof( float ), 1, fileLevel );
    fwrite( &pObj->m_fAngle, sizeof( float ), 1, fileLevel );

    ++it;
  }
  fclose( fileLevel );

}



bool CLevel::Load( DWORD dwNr )
{

  Reset();

  char      szFileName[MAX_PATH];

  wsprintf( szFileName, "data\\level%d.dat", dwNr );

  FILE*     fileLevel = fopen( szFileName, "rb" );

  if ( fileLevel == NULL )
  {
    Initialize( 100, 100 );
    return false;
  }
  fread( &m_iWidth, sizeof( m_iWidth ), 1, fileLevel );
  fread( &m_iHeight, sizeof( m_iHeight ), 1, fileLevel );

  Initialize( m_iWidth, m_iHeight );

  fread( m_pData, m_iWidth * m_iHeight * sizeof( Light::eTextureSections ), 1, fileLevel );

  size_t    iCount = 0;
  fread( &iCount, sizeof( iCount ), 1, fileLevel );

  for ( size_t i = 0; i < iCount; ++i )
  {
    Light::eGameObjectType    oType = Light::OT_INVALID;

    fread( &oType, sizeof( Light::eGameObjectType ), 1, fileLevel );

    CGameObject*    pObj = CGameObject::FromType( oType );

    if ( pObj )
    {
      fread( &pObj->m_vectPosition.x, sizeof( float ), 1, fileLevel );
      fread( &pObj->m_vectPosition.y, sizeof( float ), 1, fileLevel );
      fread( &pObj->m_fAngle, sizeof( float ), 1, fileLevel );

      pObj->m_vectPosition.x = (float)(int)pObj->m_vectPosition.x;
      pObj->m_vectPosition.y = (float)(int)pObj->m_vectPosition.y;

      m_listObjects.push_back( pObj );
    }
    else
    {
      fseek( fileLevel, sizeof( float ) * 3, SEEK_CUR );
    }
  }

  fclose( fileLevel );

  // the level is only ok if it has a player
  return !!Player();

}



CPlayer* CLevel::Player()
{

  tListObjects::iterator    it( m_listObjects.begin() );
  while ( it != m_listObjects.end() )
  {
    CGameObject*    pObj = *it;

    if ( pObj->m_Type == Light::OT_PLAYER )
    {
      return (CPlayer*)pObj;
    }
    ++it;
  }

  return NULL;

}



CGameObject* CLevel::FindObjectAt( int iX, int iY )
{

  POINT   ptPos;

  ptPos.x = iX;
  ptPos.y = iY;

  tListObjects::iterator    it( m_listObjects.begin() );
  while ( it != m_listObjects.end() )
  {
    CGameObject*    pObj = *it;

    if ( PtInRect( &pObj->CollisionRect(), ptPos ) )
    {
      return pObj;
    }
    ++it;
  }

  return NULL;

}



bool CLevel::MoveObject( CGameObject* pObj, D3DXVECTOR3 vectDelta, CGameObject*& pObjRunInto )
{

  bool bBlocked = false;

  if ( pObj )
  {
    int     iOldX = (int)pObj->m_vectPosition.x;
    int     iOldY = (int)pObj->m_vectPosition.y;

    int     iNewX = (int)( pObj->m_vectPosition.x + vectDelta.x );
    int     iNewY = (int)( pObj->m_vectPosition.y + vectDelta.y );

    while ( ( iOldX != iNewX )
    ||      ( iOldY != iNewY ) )
    {
      if ( iNewX != iOldX )
      {
        if ( IsAreaBlocked( pObj, 
                            iOldX + ( iOldX < iNewX ? 1 : -1 ) - pObj->m_rcCollision.right / 2, 
                            iOldY - pObj->m_rcCollision.bottom / 2, 
                            pObj->m_rcCollision.right, 
                            pObj->m_rcCollision.bottom,
                            pObjRunInto ) )
        {
          iNewX = iOldX;
          vectDelta.x = 0.0f;

          bBlocked = true;
        }
        else
        {
          pObj->m_vectPosition.x += ( iOldX < iNewX ? 1 : -1 );

          vectDelta.x -= ( iOldX < iNewX ? 1 : -1 );

          iOldX += ( iOldX < iNewX ? 1 : -1 );
        }
      }
      else
      {
        pObj->m_vectPosition.x += vectDelta.x;
        vectDelta.x = 0.0f;
      }
      if ( iNewY != iOldY )
      {
        if ( IsAreaBlocked( pObj,
                            iOldX - pObj->m_rcCollision.right / 2, 
                            iOldY + ( iOldY < iNewY ? 1 : -1 ) - pObj->m_rcCollision.bottom / 2, 
                            pObj->m_rcCollision.right, 
                            pObj->m_rcCollision.bottom,
                            pObjRunInto ) )
        {
          iNewY = iOldY;
          vectDelta.y = 0.0f;

          bBlocked = true;
        }
        else
        {
          pObj->m_vectPosition.y += ( iOldY < iNewY ? 1 : -1 );

          vectDelta.y -= ( iOldY < iNewY ? 1 : -1 );

          iOldY += ( iOldY < iNewY ? 1 : -1 );
        }
      }
      else
      {
        pObj->m_vectPosition.y += vectDelta.y;
        vectDelta.y = 0.0f;
      }
    }
    pObj->m_vectPosition += vectDelta;
  }

  return !bBlocked ;

}



void CLevel::CalcShadows()
{

  int   iXOffset = (int)m_fOffsetX / 32;
  int   iYOffset = (int)m_fOffsetY / 32;

  if ( m_bFullyLit )
  {
    for ( int i = 0; i < 640 / 32 + 2; ++i )
    {
      for ( int j = 0; j < 480 / 32 + 2; ++j )
      {
        tTileInfo& TileInfo = GetTileInfo( iXOffset + i, iYOffset + j );

        TileInfo.dwColor[0] = 0xffffffff;
        TileInfo.dwColor[1] = 0xffffffff;
        TileInfo.dwColor[2] = 0xffffffff;
        TileInfo.dwColor[3] = 0xffffffff;
      }
    }
    return;
  }

  // darken all
  // +2 on every border to clean the flags when the tile got scrolled out of the screen area
  for ( int i = -2; i < 640 / 32 + 4; ++i )
  {
    for ( int j = -2; j < 480 / 32 + 4; ++j )
    {
      GetTileInfo( iXOffset + i, iYOffset + j ).Darken();
    }
  }

  tListObjects::iterator    it( m_listObjects.begin() );
  while ( it != m_listObjects.end() )
  {
    CGameObject*    pObj = *it;

    if ( ( pObj->m_vectPosition.x < m_fOffsetX - 120.0f )
    ||   ( pObj->m_vectPosition.x > m_fOffsetX + 640.0f + 120.0f )
    ||   ( pObj->m_vectPosition.y < m_fOffsetY - 120.0f )
    ||   ( pObj->m_vectPosition.y > m_fOffsetY + 640.0f + 120.0f ) )
    {
      // outside the screen
      ++it;
      continue;
    }

    switch ( pObj->m_Type )
    {
      case Light::OT_PLAYER:
        if ( ( (CPlayer*)pObj )->m_bFlashLightOn )
        {
          CastLightRay( pObj->m_vectPosition, pObj->m_fAngle - 25.0f, pObj->m_fAngle + 25.0f, 240, Light::LS_PLAYER_LIGHT );
        }
        else
        {
          // fake the player visible
          GetTileInfo( (int)pObj->m_vectPosition.x / 32, (int)pObj->m_vectPosition.y / 32 ).dwLightSources |= Light::LS_PLAYER_OBJECT;
        }
        break;
      case Light::OT_GUARD:
        CastLightRay( pObj->m_vectPosition, pObj->m_fAngle - 30.0f, pObj->m_fAngle + 30.0f, 150, Light::LS_GUARD );
        break;
      case Light::OT_LAMP:
        CastLightRay( pObj->m_vectPosition, pObj->m_fAngle - 180.0f, pObj->m_fAngle + 180.0f, 
                      110.0f + 20.0f * cosf( ( (CLamp*)pObj )->m_fLightPos ),
                      Light::LS_LAMP );
        break;
      case Light::OT_WALL_LIGHT:
        CastLightRay( pObj->m_vectPosition, pObj->m_fAngle - 60.0f, pObj->m_fAngle + 60.0f, 
                      75.0f,
                      Light::LS_LAMP );
        break;
    }
    ++it;
  }

  // smoothen shadows
  for ( int i = -1; i < 640 / 32 + 2; ++i )
  {
    for ( int j = -1; j < 480 / 32 + 2; ++j )
    {
      tTileInfo&    TileInfo = GetTileInfo( iXOffset + i, iYOffset + j );

      tTileInfo&    TileInfoE = GetTileInfo( iXOffset + i + 1, iYOffset + j );
      tTileInfo&    TileInfoSE = GetTileInfo( iXOffset + i + 1, iYOffset + j + 1 );
      tTileInfo&    TileInfoS = GetTileInfo( iXOffset + i, iYOffset + j + 1 );

      float   fPower = TileInfo.fLightPower[3] 
                     + TileInfoE.fLightPower[2]
                     + TileInfoS.fLightPower[1]
                     + TileInfoSE.fLightPower[0];
      fPower *= 0.25f;

      if ( IsBlocked( iXOffset + i, iYOffset + j ) )
      {
        // can't be from the player, so the guard can't see us through the wall
        TileInfo.dwLightSources = 0;
      }

      TileInfoSE.fLightPower[0] = TileInfoS.fLightPower[1] = TileInfoE.fLightPower[2] = TileInfo.fLightPower[3] = fPower;
      TileInfo.CalcColors();
    }
  }

}



void CLevel::CastLightRay( D3DXVECTOR3 vectSource, float fStartAngle, float fEndAngle, float fMaxLength, Light::eLightSource lSource )
{

  float     fCurAngle = fStartAngle;

  D3DXVECTOR3   vectLastHitPoint = vectSource;

  if ( IsBlocked( (int)vectSource.x / 32, (int)vectSource.y / 32 ) )
  {
    // Light is inside solid!
    return;
  }

  // store lighted tiles so they get only lighted once per ray
  std::set<std::pair<int,int> >   setLightedTiles;

  SetEdgeLights( (int)vectSource.x / 32, (int)vectSource.y / 32,
                 vectSource,
                 fMaxLength );

  setLightedTiles.insert( std::make_pair( (int)vectSource.x / 32, (int)vectSource.y / 32 ) );

  while ( fCurAngle <= fEndAngle )
  {

    bool          bBlocked = false;

    D3DXVECTOR3   vectCurPos = vectSource;

    D3DXVECTOR3   vectDir;

    vectDir.x = cosf( fCurAngle * 3.1415926f / 180.0f );
    vectDir.y = -sinf( fCurAngle * 3.1415926f / 180.0f );
    vectDir.z = 0.0f;

    // find next hit tile
    //which box of the map we're in
    int iMapX = (int)vectCurPos.x / 32;
    int iMapY = (int)vectCurPos.y / 32;
    
    //length of ray from current position to next x or y-side
    float fSideDistX = 0.0f;
    float fSideDistY = 0.0f;
    
      //length of ray from one x or y-side to next x or y-side
    float fDeltaDistX = sqrt( 1 + ( vectDir.y * vectDir.y ) / ( vectDir.x * vectDir.x ) );
    float fDeltaDistY = sqrt( 1 + ( vectDir.x * vectDir.x ) / ( vectDir.y * vectDir.y ) );
    
    //what direction to step in x or y-direction (either +1 or -1)
    int iStepX = 0;
    int iStepY = 0;

    bool bHitWall = false;
    bool bNSWall  = false;

    //calculate step and initial sideDist
    if ( vectDir.x < 0 )
    {
      iStepX = -1;
      fSideDistX = ( vectCurPos.x - iMapX * 32.0f ) * fDeltaDistX;
    }
    else
    {
      iStepX = 1;
      fSideDistX = ( iMapX * 32.0f + 32.0f - vectCurPos.x ) * fDeltaDistX;
    }           
    if ( vectDir.y < 0 )
    {
      iStepY = -1;
      fSideDistY = ( vectCurPos.y - iMapY * 32.0f ) * fDeltaDistY;
    }
    else
    {
      iStepY = 1;
      fSideDistY = ( iMapY * 32.0f + 32.0f - vectCurPos.y ) * fDeltaDistY;
    }

    vectCurPos.x += fSideDistX * iStepX;
    vectCurPos.y += fSideDistY * iStepY;

    // loop DDA
    while ( !bHitWall )
    {                   
      //jump to next map square, OR in x-direction, OR in y-direction
      if ( fSideDistX < fSideDistY )
      {
        fSideDistX += fDeltaDistX * 32.0f;
        iMapX += iStepX;
        bNSWall = true;

        vectCurPos.x += 32.0f * iStepX;
        vectCurPos.y += -32.0f * sinf( fCurAngle * 3.1415926f / 180.0f );
      }
      else
      {
        fSideDistY += fDeltaDistY * 32.0f;
        iMapY += iStepY;
        bNSWall = false;

        vectCurPos.x += 32.0f * cosf( fCurAngle * 3.1415926f / 180.0f );
        vectCurPos.y += 32.0f * iStepY;
      }

      float   fDistance = D3DXVec3Length( &( vectCurPos - vectSource ) );
      if ( IsBlocked( iMapX, iMapY ) )
      {
        bHitWall = true;
        // TODO ? calc next angle
        fCurAngle += 5.0f;
      }
      //else
      {
        // apply light to the tile
        if ( setLightedTiles.find( std::make_pair( iMapX, iMapY ) ) == setLightedTiles.end() )
        {
          SetEdgeLights( iMapX, iMapY,
                         vectSource,
                         fMaxLength );

          GetTileInfo( iMapX, iMapY ).dwLightSources |= lSource;

          setLightedTiles.insert( std::make_pair( iMapX, iMapY ) );
        }
      }
      if ( fDistance >= fMaxLength + 100.0f )
      {
        // can't be visible anymore
        bHitWall = true;
        fCurAngle += 3.0f;
      }
    }  
  }

}



void CLevel::TilesInSight( D3DXVECTOR3 vectSource, float fStartAngle, float fEndAngle, float fMaxLength, 
                           std::set<std::pair<int,int> >& setTilesInSight )
{

  setTilesInSight.clear();

  float         fCurAngle = fStartAngle;

  if ( IsBlocked( (int)vectSource.x / 32, (int)vectSource.y / 32 ) )
  {
    // Light is inside solid!
    return;
  }

  // store lighted tiles so they get only lighted once per ray
  setTilesInSight.insert( std::make_pair( (int)vectSource.x / 32, (int)vectSource.y / 32 ) );

  while ( fCurAngle <= fEndAngle )
  {

    bool          bBlocked = false;

    D3DXVECTOR3   vectCurPos = vectSource;

    D3DXVECTOR3   vectDir;

    vectDir.x = cosf( fCurAngle * 3.1415926f / 180.0f );
    vectDir.y = -sinf( fCurAngle * 3.1415926f / 180.0f );
    vectDir.z = 0.0f;

    // find next hit tile
    //which box of the map we're in
    int iMapX = (int)vectCurPos.x / 32;
    int iMapY = (int)vectCurPos.y / 32;
    
    //length of ray from current position to next x or y-side
    float fSideDistX = 0.0f;
    float fSideDistY = 0.0f;
    
      //length of ray from one x or y-side to next x or y-side
    float fDeltaDistX = sqrt( 1 + ( vectDir.y * vectDir.y ) / ( vectDir.x * vectDir.x ) );
    float fDeltaDistY = sqrt( 1 + ( vectDir.x * vectDir.x ) / ( vectDir.y * vectDir.y ) );
    
    //what direction to step in x or y-direction (either +1 or -1)
    int iStepX = 0;
    int iStepY = 0;

    bool bHitWall = false;
    bool bNSWall  = false;

    //calculate step and initial sideDist
    if ( vectDir.x < 0 )
    {
      iStepX = -1;
      fSideDistX = ( vectCurPos.x - iMapX * 32.0f ) * fDeltaDistX;
    }
    else
    {
      iStepX = 1;
      fSideDistX = ( iMapX * 32.0f + 32.0f - vectCurPos.x ) * fDeltaDistX;
    }           
    if ( vectDir.y < 0 )
    {
      iStepY = -1;
      fSideDistY = ( vectCurPos.y - iMapY * 32.0f ) * fDeltaDistY;
    }
    else
    {
      iStepY = 1;
      fSideDistY = ( iMapY * 32.0f + 32.0f - vectCurPos.y ) * fDeltaDistY;
    }

    vectCurPos.x += fSideDistX * iStepX;
    vectCurPos.y += fSideDistY * iStepY;

    // loop DDA
    while ( !bHitWall )
    {                   
      //jump to next map square, OR in x-direction, OR in y-direction
      if ( fSideDistX < fSideDistY )
      {
        fSideDistX += fDeltaDistX * 32.0f;
        iMapX += iStepX;
        bNSWall = true;

        vectCurPos.x += 32.0f * iStepX;
        vectCurPos.y += -32.0f * sinf( fCurAngle * 3.1415926f / 180.0f );
      }
      else
      {
        fSideDistY += fDeltaDistY * 32.0f;
        iMapY += iStepY;
        bNSWall = false;

        vectCurPos.x += 32.0f * cosf( fCurAngle * 3.1415926f / 180.0f );
        vectCurPos.y += 32.0f * iStepY;
      }

      float   fDistance = D3DXVec3Length( &( vectCurPos - vectSource ) );
      if ( IsBlocked( iMapX, iMapY ) )
      {
        bHitWall = true;
        // TODO ? calc next angle
        fCurAngle += 5.0f;
      }
      //else
      {
        // apply light to the tile
        if ( setTilesInSight.find( std::make_pair( iMapX, iMapY ) ) == setTilesInSight.end() )
        {
          setTilesInSight.insert( std::make_pair( iMapX, iMapY ) );
        }
      }
      if ( fDistance >= fMaxLength + 100.0f )
      {
        // can't be visible anymore
        bHitWall = true;
        fCurAngle += 3.0f;
      }
    }  
  }

}



bool CLevel::IsBlocked( int iX, int iY )
{

  if ( m_pData == NULL )
  {
    return true;
  }
  if ( ( iX < 0 )
  ||   ( iX >= m_iWidth )
  ||   ( iY < 0 )
  ||   ( iY >= m_iHeight ) )
  {
    return true;
  }
  Light::eTextureSections   tsType = m_pData[iX + m_iWidth * iY];

  if ( ( tsType != Light::TS_EMPTY )
  &&   ( tsType != Light::TS_FLOOR )
  &&   ( tsType != Light::TS_MARBLE )
  &&   ( tsType != Light::TS_CHECKERBOARD ) )
  {
    return true;
  }
  return false;

}



void CLevel::SetEdgeLights( int iMapX, int iMapY, D3DXVECTOR3 vectSource, float fMaxLength )
{

  tTileInfo&    TileInfo = GetTileInfo( iMapX, iMapY );

  float   fDistance = D3DXVec3Length( &( D3DXVECTOR3( iMapX * 32.0f, iMapY * 32.0f, 0.0f ) - vectSource ) );
  if ( fDistance >= fMaxLength )
  {
    fDistance = fMaxLength;
  }
  TileInfo.fLightPower[0] += 1.0f - fDistance / fMaxLength;

  fDistance = D3DXVec3Length( &( D3DXVECTOR3( iMapX * 32.0f + 31.0f, iMapY * 32.0f, 0.0f ) - vectSource ) );
  if ( fDistance >= fMaxLength )
  {
    fDistance = fMaxLength;
  }
  TileInfo.fLightPower[1] += 1.0f - fDistance / fMaxLength;

  fDistance = D3DXVec3Length( &( D3DXVECTOR3( iMapX * 32.0f, iMapY * 32.0f + 31.0f, 0.0f ) - vectSource ) );
  if ( fDistance >= fMaxLength )
  {
    fDistance = fMaxLength;
  }
  TileInfo.fLightPower[2] += 1.0f - fDistance / fMaxLength;

  fDistance = D3DXVec3Length( &( D3DXVECTOR3( iMapX * 32.0f + 31.0f, iMapY * 32.0f + 31.0f, 0.0f ) - vectSource ) );
  if ( fDistance >= fMaxLength )
  {
    fDistance = fMaxLength;
  }
  TileInfo.fLightPower[3] += 1.0f - fDistance / fMaxLength;

}


bool CLevel::IsAreaBlocked( CGameObject* pObjToCheck, int iX, int iY, int iWidth, int iHeight, CGameObject*& pObjRunInto )
{

  for ( int i = iX / 32; i <= ( iX + iWidth - 1 ) / 32; ++i )
  {
    for ( int j = iY / 32; j <= ( iY + iHeight - 1 ) / 32; ++j )
    {
      if ( IsBlocked( i, j ) )
      {
        return true;
      }
    }
  }

  RECT    rcCheck;

  SetRect( &rcCheck, iX, iY, iX + iWidth, iY + iHeight );

  tListObjects::iterator    it( m_listObjects.begin() );
  while ( it != m_listObjects.end() )
  {
    CGameObject*    pObj = *it;

    if ( ( pObj != pObjToCheck )
    &&   ( pObj->m_Type != Light::OT_KEYCARD )
    &&   ( pObj->m_Type != Light::OT_CUPBOARD )
    &&   ( IntersectRect( &RECT(), &pObj->CollisionRect(), &rcCheck ) ) )
    {
      pObjRunInto = pObj;
      return true;
    }

    ++it;
  }

  return false;

}



void CLevel::Update( const float fElapsedTime )
{

  tListObjects::iterator    it( m_listObjects.begin() );
  while ( it != m_listObjects.end() )
  {
    CGameObject*    pObj = *it;

    pObj->Update( fElapsedTime, *this );

    ++it;
  }

  // check collisions
  CPlayer*    pPlayer = Player();
  if ( pPlayer )
  {
    tListObjects::iterator    it( m_listObjects.begin() );
    while ( it != m_listObjects.end() )
    {
      CGameObject*    pObj = *it;

      if ( pObj != pPlayer )
      {
        if ( IntersectRect( &RECT(), &pObj->CollisionRect(), &pPlayer->CollisionRect() ) )
        {
          if ( pObj->m_Type == Light::OT_KEYCARD )
          {
            // picked up keycard
            pPlayer->m_dwKeyCards++;
            delete pObj;
            it = m_listObjects.erase( it );

            theApp.m_pSoundPickup->Play( 0, 0, 0 );
            continue;
          }
          if ( pObj->m_Type == Light::OT_CUPBOARD )
          {
            // level done!
            m_bLevelDone = true;
          }
        }
      }

      pObj->Update( fElapsedTime, *this );

      ++it;
    }
  }

  if ( ( m_bPlayerHasBeenCaught )
  &&   ( !m_bCentered ) )
  {
    D3DXVECTOR3   vectDistance = m_vectCenterTarget - D3DXVECTOR3( m_fOffsetX, m_fOffsetY, 0.0f );

    //D3DXVec3Normalize( &vectDistance, &vectDistance );

    float   fDistance = D3DXVec3Length( &vectDistance );

    if ( 50.0f * fElapsedTime >= fDistance )
    {
      m_fOffsetX = m_vectCenterTarget.x;
      m_fOffsetY = m_vectCenterTarget.y;
      m_bCentered = true;
    }
    else
    {
      D3DXVec3Normalize( &vectDistance, &vectDistance );

      m_fOffsetX += vectDistance.x * 50.0f * fElapsedTime;
      m_fOffsetY += vectDistance.y * 50.0f * fElapsedTime;
    }
  }

}



bool CLevel::CanBeHeardByGuard( D3DXVECTOR3 vectPosition, float fSoundRadius )
{

  tListObjects::iterator    it( m_listObjects.begin() );
  while ( it != m_listObjects.end() )
  {
    CGameObject*    pObj = *it;

    if ( pObj->m_Type == Light::OT_GUARD )
    {
      if ( D3DXVec3Length( &( vectPosition - pObj->m_vectPosition ) ) <= fSoundRadius )
      {
        CGuard*   pGuard = (CGuard*)pObj;

        m_bFullyLit             = true;
        m_bPlayerHasBeenCaught  = true;
        m_bCentered             = false;

        m_CaughtReason          = Light::CR_HEARD;

        theApp.m_pSoundCaught->Play( 0, 0, 0 );

        // center between player and guard
        m_vectCenterTarget = ( pGuard->m_vectPosition + vectPosition ) * 0.5f - D3DXVECTOR3( 320.0f, 240.0f, 0.0f );

        pGuard->TurnToPosition( vectPosition );
        break;

        return true;
      }
    }
    ++it;
  }
  return false;

}