#include <d3d8.h>

#include ".\GSEdit.h"
#include ".\GSMenu.h"
#include ".\LightAndDark.h"
#include ".\GameObject.h"
#include ".\Util.h"



CGSEdit::CGSEdit() :
  m_iCurrentEditTile( 16 ),     // should be TS_FLOOR
  m_bTileMode( true ),
  m_dwLevel( 1 ),
  m_oType( Light::OT_PLAYER ),
  m_pCurrentObject( NULL ),
  m_pSelectedObject( NULL ),
  m_bMouseButtonReleased( true )
{

  m_vectEditTiles.push_back( Light::TS_EMPTY );
  m_vectEditTiles.push_back( Light::TS_WALL_H );
  m_vectEditTiles.push_back( Light::TS_WALL_V );
  m_vectEditTiles.push_back( Light::TS_WALL_LO );
  m_vectEditTiles.push_back( Light::TS_WALL_RO );
  m_vectEditTiles.push_back( Light::TS_WALL_LU );
  m_vectEditTiles.push_back( Light::TS_WALL_RU );
  m_vectEditTiles.push_back( Light::TS_WALL_T_S );
  m_vectEditTiles.push_back( Light::TS_WALL_T_E );
  m_vectEditTiles.push_back( Light::TS_WALL_T_N );
  m_vectEditTiles.push_back( Light::TS_WALL_T_W );
  m_vectEditTiles.push_back( Light::TS_WALL_X );
  m_vectEditTiles.push_back( Light::TS_WALL_END_E );
  m_vectEditTiles.push_back( Light::TS_WALL_END_N );
  m_vectEditTiles.push_back( Light::TS_WALL_END_W );
  m_vectEditTiles.push_back( Light::TS_WALL_END_S );
  m_vectEditTiles.push_back( Light::TS_FLOOR );
  m_vectEditTiles.push_back( Light::TS_DOOR_H );
  m_vectEditTiles.push_back( Light::TS_DOOR_V );
  m_vectEditTiles.push_back( Light::TS_MARBLE );
  m_vectEditTiles.push_back( Light::TS_CHECKERBOARD );

  m_Level.Initialize( 100, 100 );
  m_Level.Load( m_dwLevel );

  m_Level.m_bFullyLit = true;

  ShowCursor( TRUE );

  POINT   ptMouse;
  GetCursorPos( &ptMouse );

  ScreenToClient( theApp.m_hWnd, &ptMouse );
  m_iMouseX = ptMouse.x;
  m_iMouseY = ptMouse.y;

  m_pCurrentObject = CGameObject::FromType( m_oType );
  if ( m_pCurrentObject )
  {
    m_pCurrentObject->m_vectPosition.x = m_Level.m_fOffsetX + m_iMouseX;
    m_pCurrentObject->m_vectPosition.y = m_Level.m_fOffsetY + m_iMouseY;
  }

}



CGSEdit::~CGSEdit()
{

  ShowCursor( FALSE );
  if ( m_pCurrentObject )
  {
    delete m_pCurrentObject;
  }

}



void CGSEdit::Display( IDirect3DDevice8* pDevice )
{

  m_Level.Display( pDevice );

  if ( m_bTileMode )
  {
    if ( m_iCurrentEditTile < m_vectEditTiles.size() )
    {
      theApp.RenderTextureSection( m_iMouseX, m_iMouseY,
                                   theApp.m_TexSec[m_vectEditTiles[m_iCurrentEditTile]] );
    }
  }
  else
  {
    if ( m_pCurrentObject )
    {
      m_pCurrentObject->Display( pDevice, m_Level );
    }
    if ( m_pSelectedObject )
    {
      theApp.RenderTextureSectionRotated( (int)( m_pSelectedObject->m_vectPosition.x - m_Level.m_fOffsetX ),
                                          (int)( m_pSelectedObject->m_vectPosition.y - m_Level.m_fOffsetY ),
                                          theApp.m_TexSec[m_pSelectedObject->m_tsImage],
                                          m_pSelectedObject->m_fAngle,
                                          0xffff00ff );
    }
  }

  char    szTemp[MAX_PATH];
  wsprintf( szTemp, "Level: %d", m_dwLevel );
  theApp.Print( 5, 5, szTemp );

  if ( m_bTileMode )
  {
    theApp.Print( 320, 5, "Tile Mode (M to change)" );
  }
  else
  {
    theApp.Print( 320, 5, "Object Mode (M to change)" );
  }
  if ( m_Level.m_bFullyLit )
  {
    theApp.Print( 480, 5, "Lights off (L to change)" );
  }
  else
  {
    theApp.Print( 480, 5, "Lights on (M to change)" );
  }
  if ( m_pSelectedObject )
  {
    wsprintf( szTemp, "Angle: %d", (int)m_pSelectedObject->m_fAngle );
    theApp.Print( 5, 35, szTemp );
  }

  theApp.Print( 6, 421, "Cursor keys to scroll, +/- to change tile/object, </> to change level, Use Mouse to set/get", 0xff000000 );
  theApp.Print( 5, 420, "Cursor keys to scroll, +/- to change tile/object, </> to change level, Use Mouse to set/get" );

  theApp.Print( 6, 441, "1/2 Turn selected object, 3/4/5/6 set selected object to full 90 degree, F2 to save current level, A AutoWall", 0xff000000 );
  theApp.Print( 5, 440, "1/2 Turn selected object, 3/4/5/6 set selected object to full 90 degree, F2 to save current level, A AutoWall" );

  theApp.Print( 6, 461, "Cursor keys while SHIFT is down will move the selected object pixelwise", 0xff000000 );
  theApp.Print( 5, 460, "Cursor keys while SHIFT is down will move the selected object pixelwise" );

}



void CGSEdit::UpdateFrame( const float fElapsedTime )
{

  if ( !( GetAsyncKeyState( VK_SHIFT ) & 0x8000 ) )
  {
    if ( theApp.m_bKeyPressed[VK_LEFT] )
    {
      m_Level.Scroll( -520.0f * fElapsedTime, 0.0f );
    }
    if ( theApp.m_bKeyPressed[VK_RIGHT] )
    {
      m_Level.Scroll( 520.0f * fElapsedTime, 0.0f );
    }
    if ( theApp.m_bKeyPressed[VK_UP] )
    {
      m_Level.Scroll( 0.0f, -520.0f * fElapsedTime );
    }
    if ( theApp.m_bKeyPressed[VK_DOWN] )
    {
      m_Level.Scroll( 0.0f, 520.0f * fElapsedTime );
    }
  }
  if ( theApp.m_bKeyPressed['1'] )
  {
    if ( m_pSelectedObject )
    {
      m_pSelectedObject->m_fAngle -= 100.0f * fElapsedTime;
    }
  }
  if ( theApp.m_bKeyPressed['2'] )
  {
    if ( m_pSelectedObject )
    {
      m_pSelectedObject->m_fAngle += 100.0f * fElapsedTime;
    }
  }
  if ( theApp.m_bKeyPressed['3'] )
  {
    if ( m_pSelectedObject )
    {
      m_pSelectedObject->m_fAngle = 0.0f;
    }
  }
  if ( theApp.m_bKeyPressed['4'] )
  {
    if ( m_pSelectedObject )
    {
      m_pSelectedObject->m_fAngle = 90.0f;
    }
  }
  if ( theApp.m_bKeyPressed['5'] )
  {
    if ( m_pSelectedObject )
    {
      m_pSelectedObject->m_fAngle = 180.0f;
    }
  }
  if ( theApp.m_bKeyPressed['6'] )
  {
    if ( m_pSelectedObject )
    {
      m_pSelectedObject->m_fAngle = 270.0f;
    }
  }
  if ( m_pCurrentObject )
  {
    m_pCurrentObject->m_vectPosition.x = m_Level.m_fOffsetX + m_iMouseX;
    m_pCurrentObject->m_vectPosition.y = m_Level.m_fOffsetY + m_iMouseY;
  }

}



void CGSEdit::OnChar( int iChar )
{

  if ( iChar == 27 )
  {
    theApp.NextState( new CGSMenu() );
  }
  else if ( iChar == '<' )
  {
    if ( m_dwLevel > 1 )
    {
      m_dwLevel--;
      m_Level.Load( m_dwLevel );
      m_Level.m_bFullyLit = true;
      m_pSelectedObject = NULL;
    }
  }
  else if ( iChar == '>' )
  {
    if ( m_dwLevel < 99 )
    {
      m_dwLevel++;
      m_Level.Initialize( 100, 100 );
      m_Level.Load( m_dwLevel );
      m_Level.m_bFullyLit = true;
      m_pSelectedObject = NULL;
    }
  }
  else if ( iChar == '+' )
  {
    if ( m_bTileMode )
    {
      m_iCurrentEditTile = ( ( m_iCurrentEditTile + 1 ) % m_vectEditTiles.size() );
    }
    else
    {
      if ( m_pCurrentObject )
      {
        delete m_pCurrentObject;
      }
      m_oType = (Light::eGameObjectType)( ( m_oType + 1 ) % Light::OT_MAX_OBJECTS );

      m_pCurrentObject = CGameObject::FromType( m_oType );
      if ( m_pCurrentObject )
      {
        m_pCurrentObject->m_vectPosition.x = m_Level.m_fOffsetX + m_iMouseX;
        m_pCurrentObject->m_vectPosition.y = m_Level.m_fOffsetY + m_iMouseY;
      }
    }
  }
  else if ( iChar == '-' )
  {
    if ( m_bTileMode )
    {
      m_iCurrentEditTile = ( ( m_iCurrentEditTile + m_vectEditTiles.size() - 1 ) % m_vectEditTiles.size() );
    }
    else
    {
      if ( m_pCurrentObject )
      {
        delete m_pCurrentObject;
      }
      m_oType = (Light::eGameObjectType)( ( m_oType + Light::OT_MAX_OBJECTS - 1 ) % Light::OT_MAX_OBJECTS );

      m_pCurrentObject = CGameObject::FromType( m_oType );
      if ( m_pCurrentObject )
      {
        m_pCurrentObject->m_vectPosition.x = m_Level.m_fOffsetX + m_iMouseX;
        m_pCurrentObject->m_vectPosition.y = m_Level.m_fOffsetY + m_iMouseY;
      }
    }
  }
  else if ( iChar == 'm' )
  {
    m_bTileMode = !m_bTileMode;
  }
  else if ( iChar == 'l' )
  {
    m_Level.m_bFullyLit = !m_Level.m_bFullyLit;
  }
  else if ( iChar == 'a' )
  {
    AutoWall();
  }

}



void CGSEdit::OnMouse( int iX, int iY, int iButtons )
{

  m_iMouseX = iX;
  m_iMouseY = iY;

  if ( iButtons & MK_LBUTTON )
  {
    if ( m_bTileMode )
    {
      if ( m_iCurrentEditTile < m_vectEditTiles.size() )
      {
        m_Level.SetField( (int)( iX + m_Level.m_fOffsetX ) / 32,
                          (int)( iY + m_Level.m_fOffsetY ) / 32,
                          m_vectEditTiles[m_iCurrentEditTile] );
      }
    }
    else
    {
      if ( m_bMouseButtonReleased )
      {
        m_bMouseButtonReleased = false;
        CGameObject*    pObjBelow = m_Level.FindObjectAt( iX + (int)m_Level.m_fOffsetX,
                                                          iY + (int)m_Level.m_fOffsetY );
        if ( pObjBelow )
        {
          m_pSelectedObject = pObjBelow;
        }
        else if ( m_pCurrentObject )
        {
          CGameObject*    pObj = CGameObject::FromType( m_oType );
          if ( pObj )
          {
            *pObj = *m_pCurrentObject;
            m_Level.m_listObjects.push_back( pObj );
            m_pSelectedObject = pObj;
          }
        }
      }
    }
  }
  else
  {
    m_bMouseButtonReleased = true;
  }
  if ( iButtons & MK_RBUTTON )
  {
    if ( m_bTileMode )
    {
      Light::eTextureSections tsTile = m_Level.GetField( (int)( iX + m_Level.m_fOffsetX ) / 32,
                                                         (int)( iY + m_Level.m_fOffsetY ) / 32 );
      for ( size_t i = 0; i < m_vectEditTiles.size(); ++i )
      {
        if ( m_vectEditTiles[i] == tsTile )
        {
          m_iCurrentEditTile = i;
          break;
        }
      }
    }
  }

}



void CGSEdit::OnKeyDown( int iChar )
{

  switch ( iChar )
  {
    case VK_F2:
      m_Level.Save( m_dwLevel );
      break;
    case VK_DELETE:
      if ( m_pSelectedObject )
      {
        m_Level.m_listObjects.remove( m_pSelectedObject );
        delete m_pSelectedObject;
        m_pSelectedObject = NULL;
      }
      break;
    case VK_LEFT:
      if ( GetAsyncKeyState( VK_SHIFT ) & 0x8000 )
      {
        if ( m_pSelectedObject )
        {
          m_pSelectedObject->m_vectPosition.x--;
        }
      }
      break;
    case VK_RIGHT:
      if ( GetAsyncKeyState( VK_SHIFT ) & 0x8000 )
      {
        if ( m_pSelectedObject )
        {
          m_pSelectedObject->m_vectPosition.x++;
        }
      }
      break;
    case VK_UP:
      if ( GetAsyncKeyState( VK_SHIFT ) & 0x8000 )
      {
        if ( m_pSelectedObject )
        {
          m_pSelectedObject->m_vectPosition.y--;
        }
      }
      break;
    case VK_DOWN:
      if ( GetAsyncKeyState( VK_SHIFT ) & 0x8000 )
      {
        if ( m_pSelectedObject )
        {
          m_pSelectedObject->m_vectPosition.y++;
        }
      }
      break;
  }

}



void CGSEdit::AutoWall()
{

  for ( int i = 0; i < m_Level.m_iWidth; ++i )
  {
    for ( int j = 0; j < m_Level.m_iHeight; ++j )
    {
      // can tile be a wall at all?
      Light::eTextureSections   tsTile  = m_Level.GetField( i, j );
      Light::eTextureSections   tsTileN = m_Level.GetField( i, j - 1 );
      Light::eTextureSections   tsTileS = m_Level.GetField( i, j + 1 );
      Light::eTextureSections   tsTileW = m_Level.GetField( i - 1, j );
      Light::eTextureSections   tsTileE = m_Level.GetField( i + 1, j );

      if ( !Util::PotentialWall( tsTile ) )
      {
        continue;
      }
      if ( !IsFloorBeside( i, j ) )
      {
        continue;
      }

      bool      bConnectN = Util::PotentialWall( tsTileN );
      bool      bConnectS = Util::PotentialWall( tsTileS );
      bool      bConnectW = Util::PotentialWall( tsTileW );
      bool      bConnectE = Util::PotentialWall( tsTileE );

      if ( !IsFloorBeside( i, j - 1 ) )
      {
        bConnectN = false;
      }
      if ( !IsFloorBeside( i, j + 1 ) )
      {
        bConnectS = false;
      }
      if ( !IsFloorBeside( i - 1, j ) )
      {
        bConnectW = false;
      }
      if ( !IsFloorBeside( i + 1, j ) )
      {
        bConnectE = false;
      }

      if ( ( !bConnectN )
      &&   ( !bConnectW )
      &&   ( !bConnectS )
      &&   ( !bConnectE ) )
      {
        // not connected to anything
        continue;
      }
      if ( ( bConnectN )
      &&   ( bConnectW )
      &&   ( bConnectE )
      &&   ( bConnectS ) )
      {
        m_Level.SetField( i, j, Light::TS_WALL_X );
        continue;
      }
      // fugly part
      if ( bConnectN )
      {
        if ( ( bConnectW )
        &&   ( bConnectE ) )
        {
          m_Level.SetField( i, j, Light::TS_WALL_T_N );
        }
        else if ( ( bConnectW )
        &&        ( bConnectS ) )
        {
          m_Level.SetField( i, j, Light::TS_WALL_T_W );
        }
        else if ( ( bConnectE )
        &&        ( bConnectS ) )
        {
          m_Level.SetField( i, j, Light::TS_WALL_T_E );
        }
        else if ( bConnectW )
        {
          m_Level.SetField( i, j, Light::TS_WALL_RU );
        }
        else if ( bConnectE )
        {
          m_Level.SetField( i, j, Light::TS_WALL_LU );
        }
        else if ( bConnectS )
        {
          m_Level.SetField( i, j, Light::TS_WALL_V );
        }
        else
        {
          m_Level.SetField( i, j, Light::TS_WALL_END_S );
        }
      }
      else if ( bConnectS )
      {
        if ( ( bConnectW )
        &&   ( bConnectE ) )
        {
          m_Level.SetField( i, j, Light::TS_WALL_T_S );
        }
        else if ( bConnectW )
        {
          m_Level.SetField( i, j, Light::TS_WALL_RO );
        }
        else if ( bConnectE )
        {
          m_Level.SetField( i, j, Light::TS_WALL_LO );
        }
        else
        {
          m_Level.SetField( i, j, Light::TS_WALL_END_N );
        }
      }
      else if ( bConnectW )
      {
        if ( bConnectE )
        {
          m_Level.SetField( i, j, Light::TS_WALL_H );
        }
        else
        {
          m_Level.SetField( i, j, Light::TS_WALL_END_E );
        }
      }
      else
      {
        m_Level.SetField( i, j, Light::TS_WALL_END_W );
      }
    }
  }

}



bool CGSEdit::IsFloorBeside( int iX, int iY )
{

  if ( ( !Util::IsFloor( m_Level.GetField( iX - 1, iY ) ) )
  &&   ( !Util::IsFloor( m_Level.GetField( iX + 1, iY ) ) )
  &&   ( !Util::IsFloor( m_Level.GetField( iX, iY - 1 ) ) )
  &&   ( !Util::IsFloor( m_Level.GetField( iX, iY + 1 ) ) )
  &&   ( !Util::IsFloor( m_Level.GetField( iX + 1, iY + 1 ) ) )
  &&   ( !Util::IsFloor( m_Level.GetField( iX + 1, iY - 1 ) ) )
  &&   ( !Util::IsFloor( m_Level.GetField( iX - 1, iY + 1 ) ) )
  &&   ( !Util::IsFloor( m_Level.GetField( iX - 1, iY - 1 ) ) ) )
  {
    // not beside a floor tile
    return false;
  }

  return true;

}